//C5 DataFlash Reader( bulk )

#include <windows.h>
#include <winuser.h>
#include <stdio.h>
#include "c5_io.h"
#include "c5_properties.h"
#include "crc8.h"

//types
typedef unsigned char       u8;
typedef signed char         s8;
typedef unsigned short int  u16;
typedef signed short int    s16;
typedef unsigned long int   u32;
typedef signed long int     s32;

//see log_file_struct.txt
#define LOG_FILE_VERSION	(u8)0x01

#define BAUDRATE                115200 //com port baudrate
#define TIMEOUT                 2000   //com timeout, ms
#define IN_BUFFER_SIZE          16384  //com input buffer size
#define OUT_BUFFER_SIZE         512    //com output buffer size

unsigned char Com_Port_Number;

unsigned char IO_Frame[512];
unsigned char ADD, CMD, Nbyte;

FILE* file_handle;
char File_Name[] = "c5_log_file.c5l";
char File_Signature[] = "c5Logger";
char File_Comment[] = "demo;";

int Polling_Interval, Record_Counter;

void File_Close_at_Exit(void);
void C5_Close_at_Exit(void);
BOOL Save_Log_Header(void);
BOOL Save_Log_Record(void);


int main(void)
{
  printf("Enter the C5 com port number(1..255):");
  scanf("%d", &Com_Port_Number);

  if( !C5_Open( Com_Port_Number, BAUDRATE, TIMEOUT, IN_BUFFER_SIZE, OUT_BUFFER_SIZE) )
  {
    printf("Can't open port!\n");
    return 1;
  };

  //open log file
  file_handle = fopen (File_Name, "wb+");
  if(  !file_handle )
  {
    printf("Can't open file!");
    return 1;
  }

  atexit(File_Close_at_Exit);

  if(  !C5_Send_Command( C5_RX_ADDRESS, CMD_ENTER_DF_MODE, 0, IO_Frame ) )
  {
    printf("Transmite error! Command:CMD_ENTER_DF_MODE\n");
    exit(1);
  };

  if(  !C5_Get_Reply( &ADD, &CMD, &Nbyte, IO_Frame ) )
  {
    printf("Receive or CRC error!  Command:CMD_ENTER_DF_MODE\n");
    exit(1);
  };

  if( IO_Frame[0] != C5_ERR_NO )
  {
    printf("The charger should be in the root menu!\n");
    exit(1);
  };

  atexit(C5_Close_at_Exit);

  //save log header
  if( !Save_Log_Header() )
    exit(1);

  //save log
  if(  !C5_Send_Command( C5_RX_ADDRESS, CMD_DF_GET_DATA_BULK, 0, IO_Frame ) )
  {
    printf("Transmite error! Command:CMD_DF_GET_DATA_BULK");
    exit(1);
  };

  while( Save_Log_Record() ) ;

  exit(0);
}

void File_Close_at_Exit(void)
{
  fclose(file_handle);
}

void C5_Close_at_Exit(void)
{
  C5_Set_Timeout(50); //50ms

  if(  !C5_Send_Command( C5_RX_ADDRESS, CMD_DF_EXIT, 0, IO_Frame ) )
  {
    printf("Transmite error! Command:CMD_DF_EXIT\n");
    exit(1);
  };

  if(  !C5_Get_Reply( &ADD, &CMD, &Nbyte, IO_Frame ) )
  {
    printf("Receive or CRC error!  Command:CMD_DF_EXIT\n");
    exit(1);
  };

  if( IO_Frame[0] != C5_ERR_NO )
  {
    printf("Command error!  Command:CMD_DF_EXIT\n");
    exit(1);
  };

  C5_Close();
}


BOOL Save_Log_Header(void)
{
  char info[512], buff[1024], crc;
  int len_sig, len_info, len_full;
  float u_max, u_accu;
  unsigned int i_charge, i_discharge;
  unsigned int i_max, time_max, t_max;
  unsigned long day, hour, min, sec;
  long cap;

  if(  !C5_Send_Command( C5_RX_ADDRESS, CMD_DF_GET_INFO, 0, IO_Frame ) )
  {
    printf("Transmite error! Command:CMD_DF_GET_INFO\n");
    return 0;
  };

  if(  !C5_Get_Reply( &ADD, &CMD, &Nbyte, IO_Frame ) )
  {
    printf("Receive or CRC error! Command:CMD_DF_GET_INFO\n");
    return 0;
  };

  switch( IO_Frame[0] )
  {
    case  C5_DF_EMPTY:
    {
      printf("The Flash is empty!\n");
      return 0;
    }
    break;

    case  C5_DF_MANUAL_CHARGE_DATA:
    {
      Polling_Interval = (u32)IO_Frame[1];
      u_max =  (float)((u16)IO_Frame[2] | ((u16)IO_Frame[3]<<8))/1000.0;
      i_max = (u32)IO_Frame[4] | ((u32)IO_Frame[5]<<8);
      time_max = (u32)IO_Frame[6] | ((u32)IO_Frame[7]<<8) | ((u32)IO_Frame[8]<<16) | ((u32)IO_Frame[9]<<24);
      t_max = (u32)IO_Frame[10];

      sec = time_max;
      day  = sec/(24*60*60);
      hour = sec/(60*60) - day*24;
      min  = sec/60 - day*24*60 - hour*60;
      sec  = sec - day*24*60*60 - hour*60*60 - min*60;

      sprintf(info, "Program: Manual charge; Umax=%.2fV; Imax=%dmA; Max. charge time:%2dday %02dh:%02dm; Tmax=%dC; \n\rComment: %s",
              u_max,
              i_max,
              day,
              hour,
              min,
              t_max,
              File_Comment
             );
    }
    break;

    case  C5_DF_SERVICE_DATA:
    {
      Polling_Interval = (u32)IO_Frame[1];
      u_accu =  (float)((u16)IO_Frame[4] | ((u16)IO_Frame[5]<<8))/1000.0;
      cap =  (u32)IO_Frame[6] | ((u32)IO_Frame[7]<<8) | ((u32)IO_Frame[8]<<16) | ((u32)IO_Frame[9]<<24);
      i_charge = (u32)IO_Frame[10] | ((u32)IO_Frame[11]<<8);
      i_discharge = (u32)IO_Frame[12] | ((u32)IO_Frame[13]<<8);
      t_max = (u32)IO_Frame[14];
      sprintf(info, "Program: %s;  Accu: %s %.2fV/%dmAh; Ic=%dmA; Id=%dmA; Tmax=%dC;\n\rComment: %s",
              Accu_Program_Name[IO_Frame[2]-1],
              Accu_Type_Name[IO_Frame[3]-1],
              u_accu,
              cap,
              i_charge,
              i_discharge,
              t_max,
              File_Comment
             );

    }
    break;

    default:
    {
      printf("Unknow flash data type!\n");
      return 0;
    }
    break;

  }

  len_sig = strlen( File_Signature );
  len_info = strlen( info );
  if(len_info > 256)
    len_info = 256;
  memcpy( buff, File_Signature, len_sig );
  buff[len_sig] = LOG_FILE_VERSION;
  buff[len_sig+1] = len_info;                     // accu_info lenght
  buff[len_sig+2] =  CRC8_buff(buff, len_sig+2);  // crc of: signature + log-file-version + accu-info-lenght
  memcpy( &buff[len_sig+3], info, len_info );
  len_full = len_sig + 3 + len_info;
  fwrite (buff, 1, len_full, file_handle);
  crc = CRC8_buff(info, len_info);
  fwrite (&crc, 1, 1, file_handle);               //crc of accu_info

  return 1;
}


BOOL Save_Log_Record(void)
{
  unsigned long Time;
  char buff[128];

  if(  !C5_Get_Reply( &ADD, &CMD, &Nbyte, IO_Frame ) )
  {
    printf("Receive or CRC error! Command:CMD_DF_GET_DATA_BULK\n");
    return 0;
  };

  if( Nbyte == 1 )
  {
    if( Record_Counter )
    {
      printf("\nEnd of data!\nReaded records:%d\n", Record_Counter );
      return 0;
    }
    else
    {
      printf("\nThe Flash is empty!\n");
      return 0;
    }
  }

  Time = Polling_Interval*Record_Counter; //sec
  Record_Counter++;

  memcpy( buff, &Time, 4 );
  memcpy( buff+4, &IO_Frame[0], 9 );

  buff[4+9] = CRC8_buff(buff, 4+9);
  fwrite( buff, 1, 4+9+1, file_handle );
  printf("The record N%5d was stored.\r", Record_Counter);

  return 1;
}

