//Simply Pb-charger

#include <windows.h>
#include <winuser.h>
#include <stdio.h>
#include <conio.h>
#include "c5_io.h"
#include "c5_properties.h"
#include "accu_properties.h"

//types
typedef unsigned char       u8;
typedef signed char         s8;
typedef unsigned short int  u16;
typedef signed short int    s16;
typedef unsigned long int   u32;
typedef signed long int     s32;

#define POLLING_INTERVAL	1000 //ms

#define BAUDRATE                115200 //com port baudrate
#define TIMEOUT                 2000   //com timeout, ms
#define IN_BUFFER_SIZE          512    //com input buffer size
#define OUT_BUFFER_SIZE         512    //com output buffer size

unsigned char Com_Port_Number;

unsigned char IO_Frame[512];
unsigned char ADD, CMD, Nbyte;
char Error_Msg[100];

UINT idTimer;
MSG msg;

u8 Bat_Type;
u32 Bat_Capacity_mAh; //mAh

u16 Umax, Imax, Imin;
s16 Icharge;
u16 Uload, Uunload;
s8 Taccu;
u8 Remote_State;

void CALLBACK TimerProc(HWND hwnd, UINT uMsg, UINT idEvent, DWORD dwTime);
void Get_Service_Data( u8* remote_state, u16* u_load, u16* u_unload, s16* i_accu, s8* t_accu );
void Bat_Present_Test(void);
char* Get_C5_Error_Message( u8 err );
void C5_Close_at_Exit(void);
void Kill_Timer_at_Exit(void);

int main(void)
{
  //get c5 com port
  printf("Enter the C5 com port number(1..255):");
  scanf("%d", &Com_Port_Number);

  //get charge setting
  printf("Enter the PB-battery type(1: 2V; 2: 6V; 3: 12V):");
  scanf("%d", &Bat_Type);

  switch( Bat_Type )
  {
    case 1:
      Umax = PB_MAX_CELL_VOLTAGE; //mV
      break;

    case 2:
      Umax = PB_MAX_CELL_VOLTAGE*3; //mV
      break;

    case 3:
      Umax = PB_MAX_CELL_VOLTAGE*6; //mV
      break;

    default:
      //error
      printf("The battery type is wrong!\n");
      return 1;
      break;

  }

  //get battery capacity
  printf("Enter the battery capacity, mAh(milliampere/hour):");
  scanf("%d", &Bat_Capacity_mAh);

  //open c5
  if( !C5_Open( Com_Port_Number, BAUDRATE, TIMEOUT, IN_BUFFER_SIZE, OUT_BUFFER_SIZE) )
  {
    printf("Can't open port!\n");
    return 1;
  };

  //set remote control mode
  if(  !C5_Send_Command( C5_RX_ADDRESS, CMD_ENTER_RC_MODE, 0, IO_Frame ) )
  {
    printf("Transmite error! Command:CMD_ENTER_RC_MODE\n");
    return 1;
  };

  if(  !C5_Get_Reply( &ADD, &CMD, &Nbyte, IO_Frame ) )
  {
    printf("Receive or CRC error!  Command:CMD_ENTER_RC_MODE\n");
    return 1;
  };

  if( IO_Frame[0] != C5_ERR_NO )
  {
    printf("The charger should be in the root menu!\n");
    return 1;
  };

  atexit(C5_Close_at_Exit);

  //calc. Icharge max.
  Imax = Bat_Capacity_mAh/10; // set Icharge max. = 1/10 batt. capacity; mA
  if(Imax > ACCU_CURRENT_CHARGE_MAX)
    Imax = ACCU_CURRENT_CHARGE_MAX;
  if( ( (Imax * Umax)/1000000 ) > MAX_CHARGE_POWER ) //watt
  {
    //The max. charge power is exceeded --> adjust it
    Imax = (MAX_CHARGE_POWER*1000000) / Umax;
  }

  //calc. "end of charge" condition.
  Imin = (Bat_Capacity_mAh*PB_END_CURRENT)/100; //mA
  if( Imin > Imax )
    Imin = Imax / 2;

  //battery plug in test
  Bat_Present_Test();

  //start charging
  IO_Frame[0] = (u8)Umax;
  IO_Frame[1] = (u8)(Umax>>8);
  IO_Frame[2] = (u8)Imax;
  IO_Frame[3] = (u8)(Imax>>8);
  Nbyte = 4;
  if(  !C5_Send_Command( C5_RX_ADDRESS, CMD_RC_START_CHARGE, Nbyte, IO_Frame ) )
  {
    printf("Transmite error! Command:CMD_RC_START_CHARGE\n");
    exit(1);
  };

  if(  !C5_Get_Reply( &ADD, &CMD, &Nbyte, IO_Frame ) )
  {
    printf("Receive or CRC error!  Command:CMD_RC_START_CHARGE\n");
    exit(1);
  };

  if( IO_Frame[0] != C5_ERR_NO )
  {
    printf("Error: %s\n", Get_C5_Error_Message(IO_Frame[0]));
    exit(1);
  }


  //show setting info
  printf("\nSetting info: Umax.charge=%2.2fV; Imax.charge=%2.2fA; Imin.charge=%2.2fA\n\n",
        ((float)Umax)/1000, ((float)Imax)/1000, ((float)Imin)/1000);

  //start polling timer
  idTimer = SetTimer(NULL, 0, POLLING_INTERVAL, (TIMERPROC) TimerProc);
  if(  !idTimer )
  {
    printf("Can't start timer!\n");
    exit(1);
  }

  atexit(Kill_Timer_at_Exit);

  //main loop
  while(TRUE)
  {
    GetMessage(&msg, NULL, 0, 0);
    DispatchMessage(&msg);

    if( _kbhit() )
    {
      printf("Keyboard pressed...\n");
      exit(0);
    }

  }

  return 0;
}

void Bat_Present_Test(void)
{
  //batt. present test
  Get_Service_Data( &Remote_State, &Uload, &Uunload, &Icharge, &Taccu );
  if( Uunload < 200 ) // 200mV
  {
    printf("No battery!\n");
    exit(1);
  }
}

void CALLBACK TimerProc(HWND hwnd, UINT uMsg, UINT idEvent, DWORD dwTime)
{
  Get_Service_Data( &Remote_State, &Uload, &Uunload, &Icharge, &Taccu );

  switch( Remote_State )
  {
    case C5_REMOTE_NO_SERVICE:
      ;
      break;

    case C5_REMOTE_STARTING_CHARGE:
      printf("Status: Starting charge; Data: Ubat=%2.2fV; Tbat=%dC\n",
            ((float)Uunload)/1000, Taccu);
      break;

    case C5_REMOTE_STARTING_DISCHARGE:
      ;
      break;

    case C5_REMOTE_CHARGE_SERVICE:

      printf("Status: Charge; Data: Ucharge=%2.2fV; Ubat=%2.2fV; Ibat=%2.2fA; Tbat=%dC\n",
            ((float)Uload)/1000, ((float)Uunload)/1000, ((float)Icharge)/1000, Taccu);

      if( Taccu > PB_MAX_CELL_TEMP )
      {
        printf("The maximum possible temperature was reached!\n");
        exit(0);
      }

      if( Icharge < Imin )
      {
        printf("\nEnd of charging!\n\n");
        exit(0);
      }

      break;

    case C5_REMOTE_DISCHARGE_SERVICE:
      ;
      break;

    case C5_REMOTE_OVERHEATING:
        printf("Status: Overheating; Data: Ubat=%2.2fV; Tbat=%dC\n",
              ((float)Uunload)/1000, Taccu);
        printf("\nThe charger is overheating!\n\n");
        exit(1);
      break;

    default:
      printf("Unknow remote state!\n");
      exit(1);
      break;
  }



}

void Get_Service_Data( u8* remote_state, u16* u_load, u16* u_unload, s16* i_accu, s8* t_accu )
{

  if(  !C5_Send_Command( C5_RX_ADDRESS, CMD_RC_GET_SERVICE_DATA, 0, IO_Frame ) )
  {
    printf("Transmite error! Command:CMD_RC_GET_SERVICE_DATA\n");
    exit(1);
  };

  if(  !C5_Get_Reply( &ADD, &CMD, &Nbyte, IO_Frame ) )
  {
    printf("Receive or CRC error!  Command:CMD_RC_GET_SERVICE_DATA\n");
    exit(1);
  };

  *remote_state = IO_Frame[0];

  *u_load = 0;
  *u_unload = 0;
  *i_accu = 0;
  *t_accu = 0;

  switch( *remote_state )
  {
    case C5_REMOTE_NO_SERVICE:
    case C5_REMOTE_STARTING_CHARGE:
    case C5_REMOTE_STARTING_DISCHARGE:
    case C5_REMOTE_OVERHEATING:
      *u_unload = IO_Frame[1] | ( IO_Frame[2] << 8 );
      *t_accu   = IO_Frame[3];
      break;

    case C5_REMOTE_CHARGE_SERVICE:
    case C5_REMOTE_DISCHARGE_SERVICE:
      *u_load   = IO_Frame[1] | ( IO_Frame[2] << 8 );
      *u_unload = IO_Frame[3] | ( IO_Frame[4] << 8 );
      *i_accu   = IO_Frame[5] | ( IO_Frame[6] << 8 );
      *t_accu   = IO_Frame[7];
      break;

    default:
      printf("Unknow remote state!\n");
      exit(1);
      break;
  }

}

char* Get_C5_Error_Message( u8 err )
{
  switch( err )
  {
    case C5_ERR_NO:
      strcpy(Error_Msg, "C5_ERR: no errors");
      break;

    case C5_ERR_BUSY:
      strcpy(Error_Msg, "C5_ERR: device is busy");
      break;

    case C5_ERR_NOT_READY:
      strcpy(Error_Msg, "C5_ERR: device is not ready");
      break;

    case C5_ERR_PARAMETERS:
      strcpy(Error_Msg, "C5_ERR: wrong parameters");
      break;

    case C5_ERR_NO_REPLY:
      strcpy(Error_Msg, "C5_ERR: no reply");
      break;

    case C5_ERR_MAX_POWER:
      strcpy(Error_Msg, "C5_ERR: max. power is exceeded");
      break;

    case C5_ERR_OVERHEATING:
      strcpy(Error_Msg, "C5_ERR: charger is overheatig");
      break;

    default:
      strcpy(Error_Msg, "C5_ERR: unknow error");
      break;

  }

  return Error_Msg;
}

void C5_Close_at_Exit(void)
{
  C5_Set_Timeout(50); //50ms

  if(  !C5_Send_Command( C5_RX_ADDRESS, CMD_RC_EXIT, 0, IO_Frame ) )
    printf("Transmite error! Command:CMD_RC_EXIT\n");

  if(  !C5_Get_Reply( &ADD, &CMD, &Nbyte, IO_Frame ) )
    printf("Receive or CRC error!  Command:CMD_RC_EXIT\n");

  if( IO_Frame[0] != C5_ERR_NO )
    printf("Command error!  Command:CMD_RC_EXIT\n");

  C5_Close();
}

void Kill_Timer_at_Exit(void)
{
  KillTimer(NULL, idTimer);
}

