// This source code is based on open source communication protocol Wake
// http://www.caxapa.ru/lib/wake; author: Leonid I. Ridiko; e-mail: wubblick@yahoo.com
// many thanks to author

#define _C5_IO_EXPORT_

#include <windows.h>
#include <stdio.h>
#include "c5_io.h"
#include "crc8.h"

HANDLE        hCom;       //COM handle
DCB           dcb;        //COM device control block
DCB           dcbc;       //DCB copy
COMMTIMEOUTS  ComTo;      //COM timeouts
COMMTIMEOUTS  ComToc;     //COM timeouts copy

//prototypes
BOOL WINAPI AccessCOM(char *P);
BOOL WINAPI OpenCOM(char *P, DWORD baud, DWORD InQueue, DWORD OutQueue);
BOOL WINAPI CloseCOM(void);
BOOL WINAPI SetCOMReadTimeout( DWORD To );
BOOL WINAPI SetCOMWriteTimeout( DWORD To );
BOOL WINAPI SetModLns(DWORD F);
BOOL WINAPI GetModLns(LPDWORD lpD);
BOOL WINAPI PurgeCOM(void);
BOOL WINAPI FlushCOM(void);
BOOL WINAPI GetMaskCOM(LPDWORD lpEvtMask);
BOOL WINAPI SetMaskCOM(DWORD EvtMask);
BOOL WINAPI WaitEventCOM(LPDWORD lpEvtMask);
BOOL WINAPI RxByteCOM(unsigned char *b);

//---------------------------------------------------------------------------

int WINAPI DllEntryPoint(HINSTANCE hinst, unsigned long reason, void* lpReserved)
{
  return 1;
}

//----------------------- Access check for COM resource: --------------------

BOOL WINAPI AccessCOM(char *P)
{
 HANDLE hTemp = CreateFile(P,
                           GENERIC_READ | GENERIC_WRITE,
                           0,
                           NULL,
                           OPEN_EXISTING,
                           0,
                           NULL);
 if (hTemp != INVALID_HANDLE_VALUE)
 {
   CloseHandle(hTemp);
   return 1;
 }
 else
   return 0;
}

//----------------- Open COM for non overlapped operations: -----------------

BOOL WINAPI OpenCOM(char *P, DWORD baud, DWORD InQueue, DWORD OutQueue)
{
 hCom = CreateFile(P,
                   GENERIC_READ | GENERIC_WRITE,
                   0,
                   NULL,
                   OPEN_EXISTING,
                   0,
                   NULL);
 if (hCom == INVALID_HANDLE_VALUE)    return 0; //port open error
 if (!GetCommState(hCom, &dcb))       return 0; //get state error
 if (!GetCommTimeouts(hCom, &ComTo))  return 0; //get timeouts error
 dcbc = dcb; ComToc = ComTo;                    //save dcb and timeouts
 dcb.BaudRate = baud;                           //set boud rate
 dcb.ByteSize = 8;                              //set byte size
 dcb.Parity = NOPARITY;                         //set no parity
 dcb.StopBits = ONESTOPBIT;                     //set one stop bit
 dcb.fBinary = 1;                               //binary mode
 dcb.fDtrControl = DTR_CONTROL_DISABLE;         //clear DTR
 dcb.fRtsControl = RTS_CONTROL_ENABLE;          //set RTS
 if (!SetCommState(hCom, &dcb))       return 0; //set state error
 if (!SetupComm(hCom, InQueue, OutQueue)) return 0; //setup error
 ComTo.ReadIntervalTimeout = MAXDWORD;
 ComTo.ReadTotalTimeoutMultiplier = MAXDWORD;
 ComTo.ReadTotalTimeoutConstant = 1000;
 ComTo.WriteTotalTimeoutMultiplier = 0;
 ComTo.WriteTotalTimeoutConstant = 1000;
 if (!SetCommTimeouts(hCom, &ComTo))  return 0; //set timeouts error
 if (!PurgeCOM())                     return 0; //purge com error
 return 1;
}

//------------------------------ Close COM --------------------------------

BOOL WINAPI CloseCOM(void)
{
  if (!SetCommState(hCom, &dcbc))      return 0; //set state error
  if (!SetCommTimeouts(hCom, &ComToc)) return 0; //set timeouts error
  if (!CloseHandle(hCom))              return 0; //port close error
  return 1;
}

//-------------------------- Set COM Read Timeout -------------------------
BOOL WINAPI SetCOMReadTimeout( DWORD To )
{
  ComTo.ReadTotalTimeoutConstant = To;            //set timeout
  if (!SetCommTimeouts(hCom, &ComTo)) return 0;   //set timeouts error
  return 1;
}

//-------------------------- Set COM Write Timeout ------------------------
BOOL WINAPI SetCOMWriteTimeout( DWORD To )
{
  ComTo.WriteTotalTimeoutConstant = To;           //set timeout
  if (!SetCommTimeouts(hCom, &ComTo)) return 0;   //set timeouts error
  return 1;
}

//-------------------------- Set modem lines -------------------------------

BOOL WINAPI SetModLns(DWORD F)
{
  return EscapeCommFunction(hCom, F);
}

//-------------------------- Get modem lines -------------------------------

BOOL WINAPI GetModLns(LPDWORD lpD)
{
  return GetCommModemStatus(hCom, lpD);
}

//------------- Purge COM: terminates TX and RX and clears buffers ---------

BOOL WINAPI PurgeCOM(void)
{
 return PurgeComm(hCom, PURGE_TXABORT |
                        PURGE_RXABORT |
                        PURGE_TXCLEAR |
                        PURGE_RXCLEAR);
}

//--------------- Flush COM: TX buffer TX and clear then return: -----------

BOOL WINAPI FlushCOM(void)
{
  return FlushFileBuffers(hCom);
}

//------------------------- Get COM events mask ----------------------------

BOOL WINAPI GetMaskCOM(LPDWORD lpEvtMask)
{
  return GetCommMask(hCom, lpEvtMask);
}

//------------------------- Set COM events mask ----------------------------

BOOL WINAPI SetMaskCOM(DWORD EvtMask)
{
  return SetCommMask(hCom, EvtMask);
}

//--------------------------- Wait COM event -------------------------------

BOOL WINAPI WaitEventCOM(LPDWORD lpEvtMask)
{
  return WaitCommEvent(hCom, lpEvtMask, NULL);
}

//--------------------------- Read byte from COM port ----------------------
BOOL WINAPI RxByteCOM(unsigned char *b)
{
  DWORD r;
  BOOL x;

  x =  ReadFile(hCom, b, 1, &r, NULL); //read byte
  return (x && (r==1));
}

//--------------------------- Open C5 COM port -----------------------------
BOOL WINAPI C5_Open( unsigned char comport,
                     unsigned long baudrate,
                     unsigned long timeout,
                     unsigned long in_buffer_size,
                     unsigned long out_buffer_size )
{
  char com_port_name[16];

  sprintf(com_port_name,"\\\\.\\COM%d", comport);

  if( !OpenCOM( com_port_name, baudrate, in_buffer_size, out_buffer_size) )
    return 0;

  if( !SetCOMReadTimeout( timeout ) )
    return 0;

  if( !SetCOMWriteTimeout( timeout ) )
    return 0;

  return 1;
}

//---------------------------- Close C5 COM port ----------------------------
BOOL WINAPI C5_Close(void)
{
  return CloseCOM();
}

//---------------------------- Set C5 COM port timeout ----------------------
BOOL WINAPI C5_Set_Timeout( unsigned long timeout )
{
  if( !SetCOMReadTimeout( timeout ) )
    return 0;

  if( !SetCOMWriteTimeout( timeout ) )
    return 0;

  return 1;
}

//---------------------------- Purge C5 COM port ----------------------
BOOL WINAPI C5_Purge(void)
{
  return PurgeCOM();
}

//---------------------------- Flush C5 COM port ----------------------
BOOL WINAPI C5_Flush(void)
{
  return FlushCOM();
}

//--------------------------- Transmit frame -------------------------------
BOOL WINAPI C5_Send_Command(unsigned char ADDR, unsigned char CMD,
                            unsigned char N, unsigned char *Data)
{
  unsigned char Buff[518];
  unsigned char d, crc = CRC_INIT;              //init CRC
  DWORD r, j=0;
  BOOL x;

  for (int i = -4; i <= N; i++)
   {
    if ((i == -3) && (!ADDR)) i++;
    if (i == -4) d = FEND; else                 //FEND
    if (i == -3) d = ADDR; else                 //address
    if (i == -2) d = CMD;  else                 //command
    if (i == -1) d = N;    else                 //N
    if (i ==  N) d = crc;  else                 //CRC
                 d = Data[i];                   //data
    CRC8(d, &crc);
    if (i == -3) d = d | 0x80;
    if (i > -4)
     if ((d == FEND) || (d == FESC))
      { Buff[j++] = FESC;
        if (d == FEND) d = TFEND;
                  else d = TFESC;
      }
    Buff[j++] = d;
   }
  x = WriteFile(hCom, Buff, j, &r, NULL);       //TX frame
  return (x && (r==j));
}

//--------------------------- Receive frame ---------------------------------
BOOL WINAPI C5_Get_Reply(unsigned char *ADD, unsigned char *CMD,
                         unsigned char *N, unsigned char *Data)
{ int i;
  unsigned char b = 0, crc = CRC_INIT;            //init CRC

  for (i = 0; i < 512 && b != FEND; i++)
   if (!RxByteCOM(&b))     break;                 //frame synchronzation
  if (b != FEND)          return 0;               //timeout or sync error
  CRC8(b, &crc);                                  //update CRC
  *N = 0; *ADD = 0;
  for (i = -3; i<=*N; i++)
   {
    if (!RxByteCOM(&b))    break;                 //timeout error
    if (b == FESC)
     if (!RxByteCOM(&b))   break;                 //timeout error
      else
       { if (b == TFEND) b = FEND;                //TFEND <- FEND
          else
         if (b == TFESC) b = FESC;                //TFESC <- FESC
          else break;
       }
    CRC8(b, &crc);                                //update CRC
    if (i == -3)
     if ((b & 0x80) == 0) { *CMD = b; i++; }      //CMD (b.7=0)
      else *ADD = b & 0x7F;                       //ADD (b.7=1)
     else
    if (i == -2)
     if ((b & 0x80) != 0) break;                  //CMD error (b.7=1)
      else *CMD = b;                              //CMD
     else
    if (i == -1)
     *N = b;                                      //N
     else
    if (i !=  *N) Data[i] = b;                    //data
   }
  return ((i==*N+1) && !crc);                     //RX or CRC error
}

